import os
import hmac
import hashlib
import time
import uuid
import json
import re
from datetime import datetime, timezone

import requests
import qrcode
from flask import Flask, request, abort, session, redirect, url_for, render_template_string

import telebot
from telebot.types import InlineKeyboardMarkup, InlineKeyboardButton

from pymongo import MongoClient, ReturnDocument
from dotenv import load_dotenv

load_dotenv()

# ==========================================================
# AYRI CREATIVE SOLUTION - Telegram Shop Bot + Admin Panel
# Payment Gateway: TriPay (QRIS)
# Database: MongoDB
# Hosting target: cPanel (Setup Python App / Passenger WSGI)
# ==========================================================

# =========================
# Config
# =========================
STORE_NAME = os.getenv("STORE_NAME", "Ayri Creative Solution")

BOT_TOKEN = os.getenv("BOT_TOKEN", "")
TELEGRAM_WEBHOOK_SECRET = os.getenv("TELEGRAM_WEBHOOK_SECRET", "change-this-secret")
BASE_URL = os.getenv("BASE_URL", "").rstrip("/")

MONGODB_URI = os.getenv("MONGODB_URI", "")
DB_NAME = os.getenv("DB_NAME", "ayri_shop")

ADMIN_USER = os.getenv("ADMIN_USER", "admin")
ADMIN_PASS = os.getenv("ADMIN_PASS", "admin")
FLASK_SECRET_KEY = os.getenv("FLASK_SECRET_KEY", "change-me")

TRIPAY_API_KEY = os.getenv("TRIPAY_API_KEY", "")
TRIPAY_PRIVATE_KEY = os.getenv("TRIPAY_PRIVATE_KEY", "")
TRIPAY_MERCHANT_CODE = os.getenv("TRIPAY_MERCHANT_CODE", "")
TRIPAY_MODE = os.getenv("TRIPAY_MODE", "production").lower()  # production/sandbox
TRIPAY_METHOD = os.getenv("TRIPAY_METHOD", "QRIS")  # QRIS (as requested)
TRIPAY_CALLBACK_PATH = os.getenv("TRIPAY_CALLBACK_PATH", "/tripay/callback")

# Basic sanity checks (safe for cPanel)
if not BOT_TOKEN:
    raise RuntimeError("BOT_TOKEN belum di-set")
if not MONGODB_URI:
    raise RuntimeError("MONGODB_URI belum di-set")
if not BASE_URL.startswith("https://"):
    raise RuntimeError("BASE_URL harus https://domainkamu.com (Telegram webhook butuh HTTPS)")
if not TRIPAY_API_KEY or not TRIPAY_PRIVATE_KEY or not TRIPAY_MERCHANT_CODE:
    raise RuntimeError("TRIPAY_API_KEY / TRIPAY_PRIVATE_KEY / TRIPAY_MERCHANT_CODE belum di-set")

# =========================
# Flask + Telegram Bot
# =========================
app = Flask(__name__)
app.secret_key = FLASK_SECRET_KEY

bot = telebot.TeleBot(BOT_TOKEN, parse_mode="Markdown")

# =========================
# MongoDB
# =========================
mongo = MongoClient(MONGODB_URI)
db = mongo[DB_NAME]

# Collections:
# users: {user_id, username, first_name, created_at, updated_at}
# products: {product_id, name, description, price, active, created_at, updated_at}
# stock_items: {product_id, status: AVAILABLE/SOLD, payload: "email|pass", created_at, sold_at}
# orders: {order_id, user_id, chat_id, product_id, product_name, amount, status, tripay, qr_message_id, created_at, updated_at}
# transactions: {transaction_id, order_id, user_id, product_id, product_name, amount, status, details, created_at}

EMAIL_RE = re.compile(r"^[^@\s]+@[^@\s]+\.[^@\s]+$")

# =========================
# Util
# =========================
def now_utc():
    return datetime.now(timezone.utc)

def rupiah(n: int) -> str:
    return f"Rp{n:,}".replace(",", ".")

def require_admin():
    return session.get("admin") is True

def tripay_base_api():
    if TRIPAY_MODE == "sandbox":
        return "https://tripay.co.id/api-sandbox"
    return "https://tripay.co.id/api"

def hmac_sha256_hex(key: str, data: bytes) -> str:
    return hmac.new(key.encode("utf-8"), data, hashlib.sha256).hexdigest()

def tripay_signature_transaction(merchant_code: str, merchant_ref: str, amount: int) -> str:
    # Signature: HMAC-SHA256(privateKey, merchantCode + merchantRef + amount)
    sign_str = f"{merchant_code}{merchant_ref}{amount}".encode("latin-1")
    return hmac.new(TRIPAY_PRIVATE_KEY.encode("latin-1"), sign_str, hashlib.sha256).hexdigest()

def make_qr_png_bytes_from_string(qr_string: str) -> bytes:
    img = qrcode.make(qr_string)
    import io
    buf = io.BytesIO()
    img.save(buf, format="PNG")
    return buf.getvalue()

def ensure_user_saved(tg_user):
    db.users.update_one(
        {"user_id": tg_user.id},
        {"$set": {
            "user_id": tg_user.id,
            "username": tg_user.username,
            "first_name": tg_user.first_name,
            "last_name": tg_user.last_name,
            "updated_at": now_utc(),
        }, "$setOnInsert": {"created_at": now_utc()}},
        upsert=True
    )

def list_products_with_stock():
    pipeline = [
        {"$match": {"active": True}},
        {"$lookup": {
            "from": "stock_items",
            "localField": "product_id",
            "foreignField": "product_id",
            "as": "items"
        }},
        {"$addFields": {
            "stock": {
                "$size": {
                    "$filter": {"input": "$items", "as": "i", "cond": {"$eq": ["$$i.status", "AVAILABLE"]}}
                }
            }
        }},
        {"$project": {"_id": 0, "product_id": 1, "name": 1, "description": 1, "price": 1, "stock": 1}},
        {"$sort": {"name": 1}},
    ]
    return list(db.products.aggregate(pipeline))

def get_product(product_id: str):
    return db.products.find_one({"product_id": product_id, "active": True}, {"_id": 0})

def count_stock(product_id: str) -> int:
    return db.stock_items.count_documents({"product_id": product_id, "status": "AVAILABLE"})

def claim_stock_item(product_id: str):
    return db.stock_items.find_one_and_update(
        {"product_id": product_id, "status": "AVAILABLE"},
        {"$set": {"status": "SOLD", "sold_at": now_utc()}},
        return_document=ReturnDocument.AFTER
    )

# =========================
# TriPay API
# =========================
def tripay_create_transaction(order_id: str, product_name: str, amount: int, customer_name: str):
    """
    Create transaction (QRIS).
    """
    url = f"{tripay_base_api()}/transaction/create"
    signature = tripay_signature_transaction(TRIPAY_MERCHANT_CODE, order_id, amount)

    callback_url = f"{BASE_URL}{TRIPAY_CALLBACK_PATH}"

    payload = {
        "method": TRIPAY_METHOD,                 # "QRIS"
        "merchant_ref": order_id,                # our order id
        "amount": amount,
        "customer_name": customer_name or "Pelanggan",
        "customer_email": "buyer@telegram.local",
        "customer_phone": "0800000000",
        "order_items[0][name]": product_name,
        "order_items[0][price]": amount,
        "order_items[0][quantity]": 1,
        "callback_url": callback_url,
        "expired_time": int(time.time()) + 60 * 30,  # 30 minutes
        "signature": signature,
    }

    headers = {"Authorization": f"Bearer {TRIPAY_API_KEY}"}
    r = requests.post(url, data=payload, headers=headers, timeout=30)
    r.raise_for_status()
    data = r.json()
    if not data.get("success"):
        raise RuntimeError(data.get("message", "Gagal create transaksi TriPay"))
    return data["data"]

def tripay_check_status(reference: str):
    url = f"{tripay_base_api()}/transaction/check-status"
    headers = {"Authorization": f"Bearer {TRIPAY_API_KEY}"}
    r = requests.get(url, params={"reference": reference}, headers=headers, timeout=30)
    r.raise_for_status()
    return r.json()

# =========================
# Telegram Commands
# =========================
@bot.message_handler(commands=["start"])
def cmd_start(message):
    try:
        ensure_user_saved(message.from_user)
        bot.send_message(
            message.chat.id,
            f"👋 Selamat datang di *{STORE_NAME}*!\n\n"
            "Saya bisa membantu pembelian digital product dengan pembayaran *QRIS (TriPay)*.\n\n"
            "Perintah:\n"
            "• /productlist — daftar produk\n"
            "• /transactionhistory — riwayat transaksi kamu\n"
            "• /howtoorder — cara order\n"
        )
    except Exception:
        bot.send_message(message.chat.id, "Maaf, terjadi kesalahan saat onboarding. Coba lagi ya.")

@bot.message_handler(commands=["howtoorder"])
def cmd_howtoorder(message):
    bot.send_message(
        message.chat.id,
        "🧾 *Cara Order*\n\n"
        "1) Ketik /productlist\n"
        "2) Pilih produk\n"
        "3) Konfirmasi pesanan\n"
        "4) Scan QRIS dan bayar\n"
        "5) Setelah *PAID*, barang dikirim otomatis\n\n"
        "Kalau berubah pikiran, tekan tombol *Batalkan Pesanan* pada pesan QR."
    )

@bot.message_handler(commands=["productlist"])
def cmd_productlist(message):
    try:
        products = list_products_with_stock()
        if not products:
            bot.send_message(message.chat.id, "Belum ada produk tersedia saat ini.")
            return

        # Format: "Nomor. Nama - stok" (sesuai permintaan awal)
        lines = ["📦 *Daftar Produk Tersedia*\n"]
        kb = InlineKeyboardMarkup()

        for i, p in enumerate(products, start=1):
            lines.append(f"{i}. {p['name']} - {p['stock']}")
            kb.add(InlineKeyboardButton(f"{i}. {p['name']}", callback_data=f"select|{p['product_id']}"))

        bot.send_message(message.chat.id, "\n".join(lines), reply_markup=kb)
    except Exception:
        bot.send_message(message.chat.id, "Maaf, gagal memuat daftar produk. Coba lagi.")

@bot.message_handler(commands=["transactionhistory"])
def cmd_history(message):
    try:
        user_id = message.from_user.id
        txs = list(db.transactions.find({"user_id": user_id}).sort("created_at", -1).limit(20))
        if not txs:
            bot.send_message(message.chat.id, "Kamu belum punya riwayat transaksi.")
            return

        lines = ["🧾 *Riwayat Transaksi (terbaru)*\n"]
        for t in txs:
            ts = t["created_at"].astimezone(timezone.utc).strftime("%Y-%m-%d %H:%M UTC")
            lines.append(f"• `{t['order_id']}` — *{t['product_name']}* — {rupiah(t['amount'])} — *{t['status']}* — {ts}")

        bot.send_message(message.chat.id, "\n".join(lines))
    except Exception:
        bot.send_message(message.chat.id, "Maaf, gagal memuat riwayat transaksi.")

# =========================
# Telegram Callbacks
# =========================
@bot.callback_query_handler(func=lambda c: True)
def on_callback(call):
    try:
        data = call.data.split("|")
        action = data[0]

        if action == "select":
            product_id = data[1]
            product = get_product(product_id)
            if not product:
                bot.answer_callback_query(call.id, "Produk tidak ditemukan.")
                return

            stock = count_stock(product_id)
            if stock <= 0:
                bot.answer_callback_query(call.id, "Stok habis.")
                return

            desc = product.get("description", "-")

            text = (
                "✅ *Konfirmasi Pesanan*\n\n"
                f"Produk: *{product['name']}*\n"
                f"Harga: *{rupiah(int(product['price']))}*\n"
                f"Stok: *{stock}*\n\n"
                f"📝 *Deskripsi:*\n{desc}\n\n"
                "Lanjut buat pembayaran QRIS?"
            )

            kb = InlineKeyboardMarkup()
            kb.add(InlineKeyboardButton("✅ Konfirmasi", callback_data=f"confirm|{product_id}"))
            kb.add(InlineKeyboardButton("❌ Batal", callback_data="cancel_select|0"))
            bot.edit_message_text(text, call.message.chat.id, call.message.message_id, reply_markup=kb)

        elif action == "cancel_select":
            bot.edit_message_text("Dibatalkan. Silakan /productlist untuk pilih lagi.", call.message.chat.id, call.message.message_id)

        elif action == "confirm":
            product_id = data[1]
            product = get_product(product_id)
            if not product:
                bot.answer_callback_query(call.id, "Produk tidak ditemukan.")
                return

            stock = count_stock(product_id)
            if stock <= 0:
                bot.edit_message_text("❌ Maaf, stok habis.", call.message.chat.id, call.message.message_id)
                return

            order_id = uuid.uuid4().hex[:12]
            user_id = call.from_user.id
            customer_name = call.from_user.first_name or "Pelanggan"
            desc = product.get("description", "-")

            order = {
                "order_id": order_id,
                "user_id": user_id,
                "chat_id": call.message.chat.id,
                "product_id": product["product_id"],
                "product_name": product["name"],
                "amount": int(product["price"]),
                "status": "UNPAID",
                "tripay": None,
                "qr_message_id": None,
                "created_at": now_utc(),
                "updated_at": now_utc(),
            }
            db.orders.insert_one(order)

            # Create transaksi TriPay
            tripay_data = tripay_create_transaction(
                order_id=order_id,
                product_name=product["name"],
                amount=int(product["price"]),
                customer_name=customer_name
            )

            # Save tripay data
            db.orders.update_one(
                {"order_id": order_id},
                {"$set": {"tripay": tripay_data, "updated_at": now_utc()}}
            )

            # Send QR
            qr_bytes = None
            if tripay_data.get("qr_string"):
                qr_bytes = make_qr_png_bytes_from_string(tripay_data["qr_string"])
            elif tripay_data.get("qr_url"):
                resp = requests.get(tripay_data["qr_url"], timeout=30)
                resp.raise_for_status()
                qr_bytes = resp.content

            caption = (
                "🧾 *Pembayaran QRIS (TriPay)*\n\n"
                f"Order ID: `{order_id}`\n"
                f"Produk: *{product['name']}*\n"
                f"Total: *{rupiah(int(product['price']))}*\n\n"
                f"📝 *Deskripsi:*\n{desc}\n\n"
                "Silakan scan QR untuk membayar.\n"
                "Setelah pembayaran *PAID*, barang dikirim otomatis."
            )

            kb = InlineKeyboardMarkup()
            kb.add(InlineKeyboardButton("❌ Batalkan Pesanan", callback_data=f"cancel_order|{order_id}"))
            if tripay_data.get("checkout_url"):
                kb.add(InlineKeyboardButton("Buka Halaman Checkout", url=tripay_data["checkout_url"]))

            if qr_bytes:
                sent = bot.send_photo(call.message.chat.id, qr_bytes, caption=caption, reply_markup=kb)
                qr_msg_id = sent.message_id
            else:
                sent = bot.send_message(call.message.chat.id, caption, reply_markup=kb)
                qr_msg_id = sent.message_id

            db.orders.update_one(
                {"order_id": order_id},
                {"$set": {"qr_message_id": qr_msg_id, "updated_at": now_utc()}}
            )

            bot.edit_message_text(
                f"✅ Pesanan dibuat: `{order_id}`\nSaya sudah kirim QRIS-nya. Silakan bayar ya.",
                call.message.chat.id,
                call.message.message_id
            )

        elif action == "cancel_order":
            order_id = data[1]
            order = db.orders.find_one({"order_id": order_id})
            if not order or order["user_id"] != call.from_user.id:
                bot.answer_callback_query(call.id, "Order tidak ditemukan.")
                return

            if order["status"] in ("PAID", "FULFILLED"):
                bot.answer_callback_query(call.id, "Order sudah dibayar/dikirim.")
                return

            # Best effort: cek status ke TriPay sebelum cancel lokal
            try:
                ref = (order.get("tripay") or {}).get("reference")
                if ref:
                    st = tripay_check_status(ref)
                    if st.get("success") and (st.get("data") or {}).get("status") == "PAID":
                        bot.answer_callback_query(call.id, "Terdeteksi sudah PAID. Barang akan diproses.")
                        return
            except Exception:
                pass

            db.orders.update_one(
                {"order_id": order_id},
                {"$set": {"status": "CANCELED", "updated_at": now_utc()}}
            )
            db.transactions.insert_one({
                "transaction_id": uuid.uuid4().hex,
                "order_id": order_id,
                "user_id": order["user_id"],
                "product_id": order["product_id"],
                "product_name": order["product_name"],
                "amount": order["amount"],
                "status": "CANCELED",
                "details": {},
                "created_at": now_utc(),
            })

            # delete QR message best effort
            try:
                bot.delete_message(call.message.chat.id, call.message.message_id)
            except Exception:
                try:
                    bot.edit_message_caption("❌ Pesanan dibatalkan.", call.message.chat.id, call.message.message_id)
                except Exception:
                    pass

            bot.send_message(call.message.chat.id, f"❌ Pesanan `{order_id}` dibatalkan.")

        bot.answer_callback_query(call.id)

    except Exception:
        bot.answer_callback_query(call.id, "Terjadi kesalahan. Coba lagi.")
        try:
            bot.send_message(call.message.chat.id, "Maaf, sistem error. Silakan ulangi /productlist.")
        except Exception:
            pass

# =========================
# Flask Routes - Webhooks
# =========================
@app.get("/")
def home():
    return f"{STORE_NAME} - OK"

@app.get("/set-webhook")
def set_webhook():
    """
    Buka URL ini sekali setelah deploy.
    """
    url = f"{BASE_URL}/telegram/{TELEGRAM_WEBHOOK_SECRET}"
    ok = bot.set_webhook(url=url)
    return {"ok": bool(ok), "webhook": url}

@app.post("/telegram/<secret>")
def telegram_webhook(secret):
    if secret != TELEGRAM_WEBHOOK_SECRET:
        abort(403)

    try:
        update = telebot.types.Update.de_json(request.get_json(force=True))
        bot.process_new_updates([update])
        return "OK"
    except Exception:
        return "ERR", 200

# =========================
# TriPay Callback (payment_status)
# =========================
@app.post(TRIPAY_CALLBACK_PATH)
def tripay_callback():
    raw = request.get_data()
    callback_signature = request.headers.get("X-Callback-Signature", "")
    callback_event = request.headers.get("X-Callback-Event", "")

    if callback_event != "payment_status":
        return json.dumps({"success": True}), 200, {"Content-Type": "application/json"}

    local_sig = hmac_sha256_hex(TRIPAY_PRIVATE_KEY, raw)
    if not hmac.compare_digest(callback_signature, local_sig):
        return json.dumps({"success": False, "message": "Invalid signature"}), 200, {"Content-Type": "application/json"}

    data = request.get_json(force=True)
    reference = data.get("reference")
    merchant_ref = data.get("merchant_ref")  # our order id
    status = data.get("status")  # PAID/FAILED/EXPIRED/REFUND etc

    if not merchant_ref:
        return json.dumps({"success": True}), 200, {"Content-Type": "application/json"}

    order = db.orders.find_one({"order_id": merchant_ref})
    if not order:
        return json.dumps({"success": True}), 200, {"Content-Type": "application/json"}

    if order["status"] == "FULFILLED":
        return json.dumps({"success": True}), 200, {"Content-Type": "application/json"}

    # Update status order
    db.orders.update_one(
        {"order_id": merchant_ref},
        {"$set": {"status": status, "updated_at": now_utc()}}
    )

    # If PAID -> deliver stock
    if status == "PAID":
        stock_item = claim_stock_item(order["product_id"])
        if not stock_item:
            bot.send_message(
                order["chat_id"],
                f"✅ Pembayaran `{merchant_ref}` terkonfirmasi, tapi stok kosong.\n"
                "Silakan hubungi admin untuk penyelesaian ya."
            )
            db.transactions.insert_one({
                "transaction_id": uuid.uuid4().hex,
                "order_id": merchant_ref,
                "user_id": order["user_id"],
                "product_id": order["product_id"],
                "product_name": order["product_name"],
                "amount": order["amount"],
                "status": "FAILED",
                "details": {"reason": "stock_empty_after_paid", "reference": reference},
                "created_at": now_utc(),
            })
            return json.dumps({"success": True}), 200, {"Content-Type": "application/json"}

        payload = stock_item.get("payload", "")
        email, pwd = "", ""
        if "|" in payload:
            email, pwd = [x.strip() for x in payload.split("|", 1)]

        product = db.products.find_one({"product_id": order["product_id"]}, {"_id": 0})
        desc = (product or {}).get("description", "-")

        bot.send_message(
            order["chat_id"],
            "✅ *Pembayaran Berhasil!*\n\n"
            f"Order ID: `{merchant_ref}`\n"
            f"Produk: *{order['product_name']}*\n\n"
            f"📝 *Deskripsi:*\n{desc}\n\n"
            "*Detail Akun:*\n"
            f"Email: `{email}`\n"
            f"Pass: `{pwd}`\n\n"
            "Terima kasih sudah order di Ayri Creative Solution 🙏"
        )

        db.orders.update_one(
            {"order_id": merchant_ref},
            {"$set": {"status": "FULFILLED", "updated_at": now_utc()}}
        )

        db.transactions.insert_one({
            "transaction_id": uuid.uuid4().hex,
            "order_id": merchant_ref,
            "user_id": order["user_id"],
            "product_id": order["product_id"],
            "product_name": order["product_name"],
            "amount": order["amount"],
            "status": "SUCCESS",
            "details": {"reference": reference, "stock_item_id": str(stock_item.get("_id"))},
            "created_at": now_utc(),
        })

    else:
        db.transactions.insert_one({
            "transaction_id": uuid.uuid4().hex,
            "order_id": merchant_ref,
            "user_id": order["user_id"],
            "product_id": order["product_id"],
            "product_name": order["product_name"],
            "amount": order["amount"],
            "status": status,
            "details": {"reference": reference},
            "created_at": now_utc(),
        })

    return json.dumps({"success": True}), 200, {"Content-Type": "application/json"}

# =========================
# Admin Panel (minimal)
# =========================
ADMIN_BASE_HTML = """
<!doctype html>
<html>
<head>
  <meta charset="utf-8"/>
  <meta name="viewport" content="width=device-width, initial-scale=1"/>
  <title>Admin - {{store}}</title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
</head>
<body class="bg-light">
<nav class="navbar navbar-expand-lg navbar-dark bg-dark">
  <div class="container">
    <a class="navbar-brand" href="/admin">Admin {{store}}</a>
    <div class="d-flex">
      {% if admin %}
        <a class="btn btn-sm btn-outline-light me-2" href="/admin/products">Produk</a>
        <a class="btn btn-sm btn-outline-light me-2" href="/admin/stock">Stok</a>
        <a class="btn btn-sm btn-outline-light me-2" href="/admin/orders">Order</a>
        <a class="btn btn-sm btn-outline-light me-2" href="/admin/transactions">Transaksi</a>
        <a class="btn btn-sm btn-warning" href="/admin/logout">Logout</a>
      {% endif %}
    </div>
  </div>
</nav>
<div class="container py-4">
  {{body|safe}}
</div>
</body>
</html>
"""

@app.get("/admin")
def admin_index():
    if not require_admin():
        return redirect(url_for("admin_login"))
    body = """
    <div class="card">
      <div class="card-body">
        <h5 class="card-title">Dashboard</h5>
        <p class="mb-0">Kelola produk & stok, lalu bot akan jual otomatis via QRIS (TriPay).</p>
      </div>
    </div>
    """
    return render_template_string(ADMIN_BASE_HTML, store=STORE_NAME, admin=True, body=body)

@app.get("/admin/login")
def admin_login():
    body = """
    <div class="row justify-content-center">
      <div class="col-md-5">
        <div class="card">
          <div class="card-body">
            <h5 class="mb-3">Login Admin</h5>
            <form method="post" action="/admin/login">
              <div class="mb-2">
                <label class="form-label">Username</label>
                <input class="form-control" name="u" required>
              </div>
              <div class="mb-3">
                <label class="form-label">Password</label>
                <input class="form-control" type="password" name="p" required>
              </div>
              <button class="btn btn-primary w-100">Login</button>
            </form>
          </div>
        </div>
      </div>
    </div>
    """
    return render_template_string(ADMIN_BASE_HTML, store=STORE_NAME, admin=False, body=body)

@app.post("/admin/login")
def admin_login_post():
    u = request.form.get("u", "")
    p = request.form.get("p", "")
    if u == ADMIN_USER and p == ADMIN_PASS:
        session["admin"] = True
        return redirect("/admin")
    return "Login gagal", 401

@app.get("/admin/logout")
def admin_logout():
    session.pop("admin", None)
    return redirect("/admin/login")

@app.get("/admin/products")
def admin_products():
    if not require_admin():
        return redirect("/admin/login")
    products = list(db.products.find({}, {"_id": 0}).sort("name", 1))
    rows = ""
    for p in products:
        rows += f"""
        <tr>
          <td>{p.get('product_id')}</td>
          <td>{p.get('name')}</td>
          <td class="text-muted" style="max-width:420px; white-space:nowrap; overflow:hidden; text-overflow:ellipsis;">{(p.get('description') or '')}</td>
          <td>{rupiah(int(p.get('price',0)))}</td>
          <td>{'YA' if p.get('active', True) else 'TIDAK'}</td>
          <td><a class="btn btn-sm btn-outline-primary" href="/admin/products/edit/{p.get('product_id')}">Edit</a></td>
        </tr>
        """
    body = f"""
    <div class="d-flex justify-content-between align-items-center mb-3">
      <h5 class="mb-0">Produk</h5>
      <a class="btn btn-primary btn-sm" href="/admin/products/new">Tambah</a>
    </div>
    <div class="card">
      <div class="card-body">
        <table class="table table-sm">
          <thead><tr><th>ID</th><th>Nama</th><th>Deskripsi</th><th>Harga</th><th>Aktif</th><th></th></tr></thead>
          <tbody>{rows}</tbody>
        </table>
      </div>
    </div>
    """
    return render_template_string(ADMIN_BASE_HTML, store=STORE_NAME, admin=True, body=body)

@app.get("/admin/products/new")
def admin_products_new():
    if not require_admin():
        return redirect("/admin/login")
    body = """
    <div class="card"><div class="card-body">
      <h5 class="mb-3">Tambah Produk</h5>
      <form method="post" action="/admin/products/new">
        <div class="mb-2">
          <label class="form-label">Product ID (unik)</label>
          <input class="form-control" name="product_id" required>
        </div>
        <div class="mb-2">
          <label class="form-label">Nama</label>
          <input class="form-control" name="name" required>
        </div>
        <div class="mb-3">
          <label class="form-label">Deskripsi Produk</label>
          <textarea class="form-control" name="description" rows="4" placeholder="Contoh: Akun 1 bulan, garansi 7 hari, profil private..." required></textarea>
        </div>
        <div class="mb-2">
          <label class="form-label">Harga (angka)</label>
          <input class="form-control" name="price" required>
        </div>
        <div class="form-check mb-3">
          <input class="form-check-input" type="checkbox" name="active" checked>
          <label class="form-check-label">Aktif</label>
        </div>
        <button class="btn btn-primary">Simpan</button>
        <a class="btn btn-secondary" href="/admin/products">Batal</a>
      </form>
    </div></div>
    """
    return render_template_string(ADMIN_BASE_HTML, store=STORE_NAME, admin=True, body=body)

@app.post("/admin/products/new")
def admin_products_new_post():
    if not require_admin():
        return redirect("/admin/login")
    product_id = request.form.get("product_id","").strip()
    name = request.form.get("name","").strip()
    description = request.form.get("description","").strip()
    price = int(request.form.get("price","0").strip())
    active = True if request.form.get("active") == "on" else False
    db.products.update_one(
        {"product_id": product_id},
        {"$set": {
            "product_id": product_id,
            "name": name,
            "description": description,
            "price": price,
            "active": active,
            "updated_at": now_utc()
        }, "$setOnInsert": {"created_at": now_utc()}},
        upsert=True
    )
    return redirect("/admin/products")

@app.get("/admin/products/edit/<product_id>")
def admin_products_edit(product_id):
    if not require_admin():
        return redirect("/admin/login")
    p = db.products.find_one({"product_id": product_id}, {"_id": 0})
    if not p:
        return "Produk tidak ditemukan", 404
    checked = "checked" if p.get("active", True) else ""
    desc = (p.get("description","") or "").replace("<","&lt;").replace(">","&gt;")
    name = (p.get("name","") or "").replace('"', "&quot;")
    body = f"""
    <div class="card"><div class="card-body">
      <h5 class="mb-3">Edit Produk: {p.get('product_id')}</h5>
      <form method="post" action="/admin/products/edit/{p.get('product_id')}">
        <div class="mb-2">
          <label class="form-label">Nama</label>
          <input class="form-control" name="name" value="{name}" required>
        </div>
        <div class="mb-3">
          <label class="form-label">Deskripsi Produk</label>
          <textarea class="form-control" name="description" rows="4" required>{desc}</textarea>
        </div>
        <div class="mb-2">
          <label class="form-label">Harga (angka)</label>
          <input class="form-control" name="price" value="{p.get('price',0)}" required>
        </div>
        <div class="form-check mb-3">
          <input class="form-check-input" type="checkbox" name="active" {checked}>
          <label class="form-check-label">Aktif</label>
        </div>
        <button class="btn btn-primary">Simpan</button>
        <a class="btn btn-secondary" href="/admin/products">Kembali</a>
      </form>
      <hr/>
      <form method="post" action="/admin/products/delete/{p.get('product_id')}" onsubmit="return confirm('Hapus produk?');">
        <button class="btn btn-danger btn-sm">Hapus Produk</button>
      </form>
    </div></div>
    """
    return render_template_string(ADMIN_BASE_HTML, store=STORE_NAME, admin=True, body=body)

@app.post("/admin/products/edit/<product_id>")
def admin_products_edit_post(product_id):
    if not require_admin():
        return redirect("/admin/login")
    name = request.form.get("name","").strip()
    description = request.form.get("description","").strip()
    price = int(request.form.get("price","0").strip())
    active = True if request.form.get("active") == "on" else False
    db.products.update_one(
        {"product_id": product_id},
        {"$set": {"name": name, "description": description, "price": price, "active": active, "updated_at": now_utc()}}
    )
    return redirect("/admin/products")

@app.post("/admin/products/delete/<product_id>")
def admin_products_delete(product_id):
    if not require_admin():
        return redirect("/admin/login")
    db.products.delete_one({"product_id": product_id})
    return redirect("/admin/products")

@app.get("/admin/stock")
def admin_stock():
    if not require_admin():
        return redirect("/admin/login")
    products = list(db.products.find({"active": True}, {"_id": 0}).sort("name", 1))
    options = "".join([f"<option value='{p['product_id']}'>{p['name']} ({p['product_id']})</option>" for p in products])
    body = f"""
    <div class="card mb-3"><div class="card-body">
      <h5 class="mb-3">Tambah Stok (Bulk)</h5>
      <p class="text-muted mb-2">Format per baris: <code>email|pass</code></p>
      <form method="post" action="/admin/stock/add">
        <div class="mb-2">
          <label class="form-label">Produk</label>
          <select class="form-select" name="product_id" required>{options}</select>
        </div>
        <div class="mb-3">
          <label class="form-label">Isi stok</label>
          <textarea class="form-control" name="payloads" rows="10" placeholder="email1@gmail.com|pass1&#10;email2@gmail.com|pass2" required></textarea>
        </div>
        <button class="btn btn-primary">Tambah Stok</button>
      </form>
    </div></div>
    """
    return render_template_string(ADMIN_BASE_HTML, store=STORE_NAME, admin=True, body=body)

@app.post("/admin/stock/add")
def admin_stock_add():
    if not require_admin():
        return redirect("/admin/login")
    product_id = request.form.get("product_id","").strip()
    payloads = request.form.get("payloads","")
    lines = [x.strip() for x in payloads.splitlines() if x.strip()]
    docs = []
    bad = []

    for line in lines:
        if "|" not in line:
            bad.append(line)
            continue
        email, pwd = [p.strip() for p in line.split("|", 1)]
        if not email or not pwd or not EMAIL_RE.match(email):
            bad.append(line)
            continue
        docs.append({
            "product_id": product_id,
            "status": "AVAILABLE",
            "payload": f"{email}|{pwd}",
            "created_at": now_utc(),
        })

    if docs:
        db.stock_items.insert_many(docs)

    if bad:
        return ("Format salah untuk baris ini (harus email|pass):<br>" + "<br>".join(bad)), 400

    return redirect("/admin/stock")

@app.get("/admin/orders")
def admin_orders():
    if not require_admin():
        return redirect("/admin/login")
    orders = list(db.orders.find({}, {"_id": 0}).sort("created_at", -1).limit(100))
    rows = ""
    for o in orders:
        rows += f"""
        <tr>
          <td>{o.get('order_id')}</td>
          <td>{o.get('product_name')}</td>
          <td>{rupiah(int(o.get('amount',0)))}</td>
          <td>{o.get('status')}</td>
          <td>{o.get('user_id')}</td>
          <td>{(o.get('tripay') or {}).get('reference','-')}</td>
        </tr>
        """
    body = f"""
    <div class="card"><div class="card-body">
      <h5 class="mb-3">Order (100 terbaru)</h5>
      <table class="table table-sm">
        <thead><tr><th>Order ID</th><th>Produk</th><th>Amount</th><th>Status</th><th>User</th><th>Ref</th></tr></thead>
        <tbody>{rows}</tbody>
      </table>
    </div></div>
    """
    return render_template_string(ADMIN_BASE_HTML, store=STORE_NAME, admin=True, body=body)

@app.get("/admin/transactions")
def admin_transactions():
    if not require_admin():
        return redirect("/admin/login")
    txs = list(db.transactions.find({}, {"_id": 0}).sort("created_at", -1).limit(200))
    rows = ""
    for t in txs:
        rows += f"""
        <tr>
          <td>{t.get('order_id')}</td>
          <td>{t.get('product_name')}</td>
          <td>{rupiah(int(t.get('amount',0)))}</td>
          <td>{t.get('status')}</td>
          <td>{t.get('user_id')}</td>
        </tr>
        """
    body = f"""
    <div class="card"><div class="card-body">
      <h5 class="mb-3">Transaksi (200 terbaru)</h5>
      <table class="table table-sm">
        <thead><tr><th>Order ID</th><th>Produk</th><th>Amount</th><th>Status</th><th>User</th></tr></thead>
        <tbody>{rows}</tbody>
      </table>
    </div></div>
    """
    return render_template_string(ADMIN_BASE_HTML, store=STORE_NAME, admin=True, body=body)
